/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.intrusiondetection
{

    using org.owasp.appsensor;
    using System;
    using System.Collections.Generic;

/**
 * This interface represents the collection of intrusions for a specific user.
 * These records are managed by and housed in the intrusion store.  
 * <p>
 * The essential ways of interacting with an instantiation of this 
 * interface once it is created
 * are to add intrusions and violations as events/attacks occur and to 
 * query the class for information about intrusions and violations 
 * that have occurred. 
 * 
 * jtm - 9/16/2010 - Updated this from a class to an interface in order 
 * to support different implementations.
 * 
 * @author Michael Coates (michael.coates .at. owasp.org) 
 *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton (jtmelton .at. gmail.com)
 *         <a href="http://www.jtmelton.com/">jtmelton</a>
 * @since February 24, 2010
 */
public interface IntrusionRecord {
		
	/**
	 * This method adds the specified intrusion to the set 
	 * of intrusions for this user.
	 * @param aie intrusion to add
	 */
	void AddIntrusionToRecord(AppSensorIntrusion aie);

	/**
	 * This method returns the number of all IntrusionExceptions that have
	 * been thrown by the app regardless of subclass.
	 * 
	 * @return number of IntrusionExceptions
	 */
	int GetNumberOfAllIntrusions();
	
	/**
	 * This method gets the number of times a given "intrusion" has been 
	 * committed.
	 * <ul>  
	 * <li>In the case that it is not appsensor specific, then the eventCode (exception class name) 
	 * is checked, and only those types of issues are considered in the count.
	 * </li><li>
	 * In the case that it is appsensor specific, then the eventCode (detection point)
	 * is checked, and only those types of issues are considered in the count.
	 * </li><li>
	 * Finally, there is an "ALL_INTRUSIONS" option that can be passed, and that 
	 * considers all exceptions, regardless of source.
	 * </li>
	 * </ul>
	 * 
	 * @param eventCode
	 * 			App-Sensor specific eventCode, or "non-appsensor" identifier or "all" identifier
	 * @return number of IntrusionExceptions
	 */
	int GetNumberOfIntrusions(String eventCode);

	/**
	 * Getter for user id
	 * @return user id
	 */
    String UserID { get; }

	/**
	 * Add a violation to the current intrusion record for this user.  
	 * This is called by the IntrusionDetector when the specified 
	 * threshold has been reached.  
	 * @param eventCode event code for the violation that has just been triggered
	 */
	void AddViolation(String eventCode);

	/**
	 * Getter for last violation
	 * @return last violation
	 */
	String GetLastViolation();

	/**
	 * Setter for last response action for a given eventcode
	 * @param lastResponseAction last response action
	 * @param eventCode the event code that this response is for
	 */
	void SetLastResponseAction(String lastResponseAction, String eventCode);
	
	/**
	 * Getter for last response action for the specified event code
	 * @param eventCode event code to retrieve last response action for
	 * @return last response action
	 */
	String GetLastResponseAction(String eventCode);

	/**
	 * Return the number of intrusions found for the current user matching the 
	 * specified event code that have occurred in the most recent interval 
	 * specified in seconds
	 * @param eventCode event code to match on intrusions
	 * @param intervalInSeconds number of seconds to go back from current time
	 * 			to look for matching intrusions
	 * @return number of intrusions found for the current user matching the 
	 * 			specified event code that have occurred in the most recent 
	 * 			interval specified in seconds
	 */
	int GetNumberOfIntrusionsInInterval(String eventCode, long intervalInSeconds);

	/**
	 * Return a copy of the intrusion collection
	 * @return copy of the intrusion collection
	 */
	List<AppSensorIntrusion> GetCopyIntrusionsObserved();
} 
}